/*=============================================================================
	IpDrvPrivate.h: Unreal TCP/IP driver.
	Copyright 1997-1999 Epic Games, Inc. All Rights Reserved.

Revision history:
	* Created by Tim Sweeney.
=============================================================================*/

#ifndef UNIPDRV_H
#define UNIPDRV_H

#if _MSC_VER
	#pragma warning( disable : 4201 )
#endif

// Socket API.
#if _MSC_VER
	#define __WINSOCK__ 1
	#define SOCKET_API TEXT("WinSock")
#else
	#define __BSD_SOCKETS__ 1
	#if ((MACOSX) || (defined __FreeBSD__))
		#define __BSD_SOCKETS_TRUE_BSD__ 1
	#endif
	#define SOCKET_API TEXT("Sockets")
#endif

// WinSock includes.
#if __WINSOCK__
#pragma pack(push,8)
#ifdef _XBOX
	#include <xtl.h>
#else
	#include <winsock2.h>
	#include <ws2tcpip.h>
	#include <windows.h>
	#include <conio.h>
#endif
#pragma pack(pop)
#endif

// BSD socket includes.
#if __BSD_SOCKETS__
	#include <stdio.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <netdb.h>
	#include <sys/uio.h>
	#include <sys/ioctl.h>
	#include <sys/time.h>
	#include <errno.h>
	#include <pthread.h>
	#include <fcntl.h>

	// Handle glibc < 2.1.3
	#ifndef MSG_NOSIGNAL
	#define MSG_NOSIGNAL 0x4000
	#endif
#endif

#include <thread>

/*----------------------------------------------------------------------------
	API.
----------------------------------------------------------------------------*/

#ifndef IPDRV_API
	#define IPDRV_API DLL_IMPORT
#endif

/*-----------------------------------------------------------------------------
	Includes..
-----------------------------------------------------------------------------*/

#include "Core.h"
#include "Engine.h"
#include "UnNet.h"
#include "UnSocket.h"

/*-----------------------------------------------------------------------------
	Definitions.
-----------------------------------------------------------------------------*/

// Globals.
IPDRV_API extern UBOOL GIpDrvInitialized;

/*-----------------------------------------------------------------------------
	More Includes.
-----------------------------------------------------------------------------*/

#include "InternetLink.h"

/*-----------------------------------------------------------------------------
	Host resolution thread.
-----------------------------------------------------------------------------*/

IPDRV_API void ResolveThreadEntry( FResolveInfo* Arg );

//
// Class for creating a background thread to resolve a host.
//
class FResolveInfo
{
public:
	// Variables.
	FIpAddr Addr;
	FString HostName;
	FString Error;
	std::thread Thread;

	// Functions.
	FResolveInfo( const TCHAR* InHostName )
	{	
		debugf( TEXT("Resolving %s..."), InHostName );

		HostName = InHostName;
		Thread = std::thread{ ResolveThreadEntry, this };

	}
	~FResolveInfo() {
		Thread.detach();
	}
	UBOOL Resolved()
	{
		if (Thread.get_id() == std::thread::id{}) {
			Thread.join();
			return true;
		}

		return false;
	}
	const TCHAR* GetError()
	{
		return *Error;
	}
	const in_addr GetAddr()
	{
		return Addr.Addr;
	}
	const TCHAR* GetHostName()
	{
		return *HostName;
	}
};

/*-----------------------------------------------------------------------------
	Bind to next available port.
-----------------------------------------------------------------------------*/

//
// Bind to next available port.
//
inline int bindnextport( SOCKET s, struct sockaddr* addr, socklen_t addrlen, int portcount, int portinc )
{
	guard(bindnextport);

#ifdef __linux__
    // Long explanation follows:
    //
    // Using port 0 tells the OS to allocate an unused port. But, this will
    // allocate the same port as the socket we just closed on Linux's
    // TCP/IP stack. If you get a stray packet from the last socket, the game
    // gets VERY confused. This is most obvious by hitting a passworded server
    // with the wrong password. We connect, fail to gain access, user enters a
    // password and we closesocket() and immediately hit the server again with
    // a newly bound port...and gets some extra unexpected chatter that hangs
    // the client.
    //
    // The workaround is to pick our own random ports if port zero is
    // requested. If we can't find a valid port above 1024 (below 1024 is
    // reserved for the superuser) after a couple tries, we block for three
    // seconds and then continue as normal...the blocking sucks, but
    // tends to fix things, too. Oh well.
    //
    // The MacOS X and Win32 TCP/IP stacks do not exhibit this problem.
    //
    //    --ryan.

    if( addr->sin_port==0 )
    {
        // try 10 random ports...
        for (int i = 0; i < 10; i++)
        {
            short p = ((short) (31742.0f * rand() / (RAND_MAX + 1.0))) + 1024;
            addr->sin_port = htons(p);
            int rc = bindnextport(s, addr, portcount, portinc);
            if (rc != 0)
                return(rc);  // got one.
        }

        debugf( NAME_Warning, TEXT("Failed to manually bind random port. Letting OS do it instead.") );
        appSleep(3.0f);
        addr->sin_port = 0;
    }
#endif


    // sockaddr_in and sockaddr_in6 share a common initial sequence that
    // includes the port
	sockaddr_in* addr_in = (sockaddr_in*)addr;
	for( int i=0; i<portcount; i++ )
	{
		if( bind( s, addr, addrlen ) == 0 )
		{
			if (ntohs(addr_in->sin_port) != 0)
				return ntohs(addr_in->sin_port);
			else
			{
				// 0 means allocate a port for us, so find out what that port was
				sockaddr_storage boundaddr;
				socklen_t boundaddrlen = sizeof(boundaddr);
				getsockname ( s, (sockaddr*)(&boundaddr), &boundaddrlen);
				addr_in = (sockaddr_in*)&boundaddr;
				return ntohs(addr_in->sin_port);
			}
		}
		if (addr_in->sin_port==0 )
			break;
		addr_in->sin_port = htons( ntohs(addr_in->sin_port) + portinc );
	}
	return 0;
	unguard;
}

inline int getlocalhostaddr( FOutputDevice& Out, in_addr &HostAddr )
{
	guard(getlocalhostaddr);
	int CanBindAll = 0;
	IpSetInt( HostAddr, INADDR_ANY );
	TCHAR Home[256]=TEXT("");
	if( Parse(appCmdLine(),TEXT("MULTIHOME="),Home,ARRAY_COUNT(Home)) )
	{
		TCHAR *A, *B, *C, *D;
		A=Home;
		if
		(	(A=Home)!=NULL
		&&	(B=appStrchr(A,'.'))!=NULL
		&&	(C=appStrchr(B+1,'.'))!=NULL
		&&	(D=appStrchr(C+1,'.'))!=NULL )
		{
			IpSetBytes( HostAddr, appAtoi(A), appAtoi(B+1), appAtoi(C+1), \
				appAtoi(D+1) );
		}
		else Out.Logf( TEXT("Invalid multihome IP address %s"), Home );
	}
	else
    {
        TCHAR HostName[256] = TEXT("");
        if (IpDrvGetHostName(HostName, ARRAY_COUNT(HostName)))
            Out.Logf(TEXT("%s: gethostname failed (%s)"), SOCKET_API, SocketError());

        ADDRINFOT Hint{};
        Hint.ai_family = AF_INET;

        ADDRINFOT* AddrInfo;
        auto Result = GetAddrInfo(HostName, NULL, &Hint, &AddrInfo);
        if (Result != 0)
            Out.Logf(TEXT("GetAddrInfo failed (%s)"), SocketError());

		if (AddrInfo) {
			HostAddr = ((sockaddr_in*)AddrInfo->ai_addr)->sin_addr;
            if (!ParseParam(appCmdLine(), TEXT("PRIMARYNET")))
                CanBindAll = 1;
            static UBOOL First = 0;
            if (!First) {
                First = 1;
                debugf(NAME_Init, TEXT("%s: I am %s (%s)"), SOCKET_API, HostName, *IpString(HostAddr));
            }
		} else {
			Out.Logf(TEXT("GetAddrInfo failed (%s) ... can't get local IP."), SocketHostError());
		}
	}
	return CanBindAll;
	unguard;
}

//
// Get local IP to bind to
//
inline FIpAddr getlocalbindaddr( FOutputDevice& Out )
{
	guard(getlocalbindaddr);

	ADDRINFOT Hint{};
	Hint.ai_family = AF_UNSPEC;
	Hint.ai_flags = AI_PASSIVE | AI_ADDRCONFIG;

	ADDRINFOT* AddrInfo;
	auto Error = GetAddrInfo(TEXT(""), nullptr, &Hint, &AddrInfo);
	if (Error || AddrInfo == nullptr) {
		Out.Logf(TEXT("Error finding address to bind to: %s"), SocketError());
		return FIpAddr{};
	}

    FIpAddr Result{ AddrInfo->ai_addr };

    FreeAddrInfo(AddrInfo);
    return Result;

	unguard;
}
/*-----------------------------------------------------------------------------
	Public includes.
-----------------------------------------------------------------------------*/

#include "IpDrvClasses.h"
#include "HTTPDownload.h"

#if __STATIC_LINK
#define NAMES_ONLY
#define NATIVE_DEFS_ONLY
#define AUTOGENERATE_NAME(name)
#define AUTOGENERATE_FUNCTION(cls,idx,name)
#include "IpDrvClasses.h"
#undef AUTOGENERATE_FUNCTION
#undef AUTOGENERATE_NAME
#undef NATIVE_DEFS_ONLY
#undef NAMES_ONLY
#endif

#endif // UNIPDRV_H

/*-----------------------------------------------------------------------------
	The End.
-----------------------------------------------------------------------------*/

